<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

$roleName = $_SESSION['role_name'] ?? '';
$userId = (int)($_SESSION['user_id'] ?? 0);
$canCreate = in_array($roleName, ['Agent Marketing', 'Chef Marketing', 'Accountant', 'Caissier_Comptable'], true);
$canViewAll = in_array($roleName, ['DG', 'DGA', 'Admin', 'Board', 'Operation Manager'], true);
if (!$canCreate && !$canViewAll) {
    header("Location: /masunzu_bar_hotel/dashboards/unauthorized.php");
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    die("<p style='color:#B00020;text-align:center;padding:20px;'>Connexion BD impossible.</p>");
}

$successMessage = '';
$errors = [];

// Fetch regions and depots
$regions = [];
$regionResult = $conn->query("SELECT id, name FROM region ORDER BY name");
if ($regionResult instanceof mysqli_result) {
    $regions = $regionResult->fetch_all(MYSQLI_ASSOC);
    $regionResult->free();
}
$depots = [];
$depotResult = $conn->query("SELECT id, name FROM province ORDER BY name");
if ($depotResult instanceof mysqli_result) {
    $depots = $depotResult->fetch_all(MYSQLI_ASSOC);
    $depotResult->free();
}

if ($canCreate && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_observation'])) {
    $subject = trim($_POST['subject'] ?? '');
    $details = trim($_POST['details'] ?? '');
    $regionId = filter_input(INPUT_POST, 'region_id', FILTER_VALIDATE_INT, ['options' => ['default' => null]]);
    $provinceId = filter_input(INPUT_POST, 'province_id', FILTER_VALIDATE_INT, ['options' => ['default' => null]]);

    if ($subject === '') {
        $errors[] = "Le sujet est requis.";
    }
    if ($details === '') {
        $errors[] = "Merci de décrire l'observation.";
    }

    if (empty($errors)) {
        $stmt = $conn->prepare("INSERT INTO field_observation (author_id, region_id, province_id, subject, details) VALUES (?, ?, ?, ?, ?)");
        if ($stmt) {
            $stmt->bind_param('iiiss', $userId, $regionId, $provinceId, $subject, $details);
            if ($stmt->execute()) {
                $successMessage = "Observation enregistrée.";
                // Notify admins and operation managers
                $notifyRoles = ['Admin', 'Operation Manager'];
                $escaped_roles = array_map(fn($roleName) => "'" . $conn->real_escape_string($roleName) . "'", $notifyRoles);
                $roleList = implode(', ', $escaped_roles);
                if (!empty($roleList)) {
                    $roleStmt = $conn->prepare("SELECT u.id FROM user u JOIN role r ON u.role_id = r.id WHERE r.name IN ($roleList)");
                    if ($roleStmt) {
                        $roleStmt->execute();
                        $roleResult = $roleStmt->get_result();
                        while ($row = $roleResult->fetch_assoc()) {
                            $notifyMessage = sprintf(
                                'Nouvelle observation terrain : %s',
                                mb_substr($subject, 0, 60)
                            );
                            mbh_notify_user($conn, (int)$row['id'], $notifyMessage, '/masunzu_bar_hotel/modules/customers/observations.php');
                        }
                        $roleStmt->close();
                    }
                }
            } else {
                $errors[] = "Impossible d'enregistrer l'observation.";
            }
            $stmt->close();
        } else {
            $errors[] = "Erreur: " . $conn->error;
        }
    }
}

$observations = [];
$obsQuery = "
    SELECT fo.id, fo.subject, fo.details, fo.created_at,
           r.name AS region_name,
           p.name AS depot_name,
           u.full_name AS author_name
    FROM field_observation fo
    LEFT JOIN region r ON fo.region_id = r.id
    LEFT JOIN province p ON fo.province_id = p.id
    LEFT JOIN user u ON fo.author_id = u.id
";
if (!$canViewAll) {
    $obsQuery .= " WHERE fo.author_id = ? ";
}
$obsQuery .= " ORDER BY fo.created_at DESC";
$stmtObs = $conn->prepare($obsQuery);
if ($stmtObs) {
    if (!$canViewAll) {
        $stmtObs->bind_param('i', $userId);
    }
    $stmtObs->execute();
    $resultObs = $stmtObs->get_result();
    $observations = $resultObs->fetch_all(MYSQLI_ASSOC);
    $stmtObs->close();
}
$conn->close();

?>
<?php include __DIR__ . '/../../includes/header.php'; ?>
<div style="padding: 20px;">
    <div style="background: #F9F5EF; border-radius: 12px; padding: 20px; box-shadow: 0 15px 35px rgba(75,47,31,0.1);">
        <h1 style="color:#4B2F1F;">Observations terrain</h1>
        <p style="color:#6D4C41;">
            Les agents commerciaux et financiers consignent ici leurs observations. Les dirigeants disposent d'une visibilité complète.
        </p>

        <?php if (!empty($errors)): ?>
            <div style="background:#FFEBEE;color:#C62828;padding:12px;border-radius:8px;margin-bottom:15px;">
                <?php foreach ($errors as $error): ?>
                    <div><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
        <?php if ($successMessage): ?>
            <div style="background:#E8F5E9;color:#2E7D32;padding:12px;border-radius:8px;margin-bottom:15px;">
                <?php echo htmlspecialchars($successMessage, ENT_QUOTES, 'UTF-8'); ?>
            </div>
        <?php endif; ?>

        <?php if ($canCreate): ?>
        <form method="post" style="margin-top:20px; display:grid; grid-template-columns: repeat(auto-fit, minmax(260px,1fr)); gap:15px;">
            <div style="grid-column: span 2;">
                <label style="display:block;color:#6D4C41;font-weight:600;">Sujet</label>
                <input type="text" name="subject" required style="width:100%;padding:10px;border:1px solid #D7CCC8;border-radius:8px;">
            </div>
            <div>
                <label style="display:block;color:#6D4C41;font-weight:600;">Région</label>
                <select name="region_id" style="width:100%;padding:10px;border:1px solid #D7CCC8;border-radius:8px;">
                    <option value="">-- Indifférent --</option>
                    <?php foreach ($regions as $region): ?>
                        <option value="<?php echo (int)$region['id']; ?>"><?php echo htmlspecialchars($region['name'], ENT_QUOTES, 'UTF-8'); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div>
                <label style="display:block;color:#6D4C41;font-weight:600;">Dépôt</label>
                <select name="province_id" style="width:100%;padding:10px;border:1px solid #D7CCC8;border-radius:8px;">
                    <option value="">-- Non spécifié --</option>
                    <?php foreach ($depots as $depot): ?>
                        <option value="<?php echo (int)$depot['id']; ?>"><?php echo htmlspecialchars($depot['name'], ENT_QUOTES, 'UTF-8'); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div style="grid-column: span 2;">
                <label style="display:block;color:#6D4C41;font-weight:600;">Observation</label>
                <textarea name="details" rows=4 required style="width:100%;padding:10px;border:1px solid #D7CCC8;border-radius:8px;"></textarea>
            </div>
            <div style="grid-column: span 2; text-align:right;">
                <button type="submit" name="create_observation" value="1" style="background:#7C4A33;color:#FFF;border:none;padding:10px 24px;border-radius:8px;cursor:pointer;">
                    Enregistrer
                </button>
            </div>
        </form>
        <?php endif; ?>

        <div style="margin-top:30px; display:flex; justify-content:space-between; align-items:center; flex-wrap:wrap; gap:10px;">
            <div style="font-weight:600;color:#6D4C41;">Total: <?php echo count($observations); ?> observation<?php echo count($observations) > 1 ? 's' : ''; ?></div>
            <input id="observation-search" type="search" placeholder="Rechercher sujet, auteur ou dépôt..." style="padding:8px 12px; border-radius:8px; border:1px solid #C0B4A6; min-width:200px;">
        </div>
        <div style="margin-top:10px; overflow-x:auto;">
            <table style="width:100%; border-collapse:collapse;">
                <thead>
                    <tr style="background:#7C4A33;color:#FFF;">
                        <th style="padding:10px;text-align:left;">Sujet</th>
                        <th style="padding:10px;text-align:left;">Auteur</th>
                        <th style="padding:10px;text-align:left;">Région / Dépôt</th>
                        <th style="padding:10px;text-align:left;">Créé le</th>
                        <th style="padding:10px;text-align:left;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                <?php if (empty($observations)): ?>
                    <tr>
                        <td colspan="5" style="padding:14px;text-align:center;color:#6D4C41;">Aucune observation disponible.</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($observations as $observation): ?>
                        <tr class="observation-row" data-search="<?php echo htmlspecialchars(strtolower(($observation['subject'] ?? '') . ' ' . ($observation['author_name'] ?? '') . ' ' . ($observation['region_name'] ?? '') . ' ' . ($observation['depot_name'] ?? '')), ENT_QUOTES, 'UTF-8'); ?>" style="border-bottom:1px solid #E0E0E0;">
                            <td style="padding:12px;"><?php echo htmlspecialchars($observation['subject'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding:12px;"><?php echo htmlspecialchars($observation['author_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding:12px;">
                                <?php echo htmlspecialchars($observation['region_name'] ?? '—', ENT_QUOTES, 'UTF-8'); ?><br>
                                <span style="color:#8D6E63;"><?php echo htmlspecialchars($observation['depot_name'] ?? 'Non défini', ENT_QUOTES, 'UTF-8'); ?></span>
                            </td>
                            <td style="padding:12px;"><?php echo htmlspecialchars(date('d/m/Y H:i', strtotime($observation['created_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding:12px;">
                                <button class="view-observation" data-details="<?php echo htmlspecialchars(json_encode([
                                        'subject' => $observation['subject'] ?? '',
                                        'author' => $observation['author_name'] ?? '',
                                        'region' => $observation['region_name'] ?? '',
                                        'depot' => $observation['depot_name'] ?? '',
                                        'created_at' => $observation['created_at'] ?? '',
                                        'details' => $observation['details'] ?? '',
                                        'id' => $observation['id'] ?? 0,
                                    ], JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP), ENT_QUOTES, 'UTF-8'); ?>" style="background:#FFFFFF;border:1px solid #7C4A33;color:#7C4A33;padding:6px 12px;border-radius:6px;cursor:pointer;">Voir</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>
            </table>
        </div>
        <div id="observation-modal" style="display:none;position:fixed;inset:0;background:rgba(0,0,0,0.55);align-items:center;justify-content:center;padding:20px;z-index:1000;">
            <div style="background:#FDFBF7;border-radius:16px;max-width:600px;width:100%;padding:24px;position:relative;box-shadow:0 15px 45px rgba(0,0,0,0.4);">
                <button id="close-observation-modal" style="position:absolute;right:16px;top:16px;background:none;border:none;font-size:24px;color:#4B2F1F;cursor:pointer;">×</button>
                <h2 id="modal-subject" style="margin-top:0;color:#331E10;"></h2>
                <p id="modal-meta" style="color:#6D4C41;font-size:14px;margin:6px 0 12px;"></p>
                <div id="modal-details" style="color:#332318;line-height:1.6;white-space:pre-line;"></div>
            </div>
        </div>
    </div>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
<script>
    (function() {
        const searchInput = document.getElementById('observation-search');
        const rows = document.querySelectorAll('.observation-row');
        const normalize = value => value.toString().normalize('NFD').replace(/[\u0300-\u036f]/g, '').toLowerCase();
        searchInput.addEventListener('input', () => {
            const needle = normalize(searchInput.value.trim());
            rows.forEach(row => {
                const haystack = normalize(row.dataset.search || '');
                row.style.display = haystack.includes(needle) || needle === '' ? '' : 'none';
            });
        });

        const modal = document.getElementById('observation-modal');
        const closeModal = document.getElementById('close-observation-modal');
        const meta = document.getElementById('modal-meta');
        const subject = document.getElementById('modal-subject');
        const details = document.getElementById('modal-details');

        document.querySelectorAll('.view-observation').forEach(button => {
            button.addEventListener('click', () => {
                const payload = JSON.parse(button.dataset.details);
                subject.textContent = payload.subject || 'Observation';
                meta.textContent = `Auteur : ${payload.author || 'N/A'} · Région : ${payload.region || 'N/A'} · Dépôt : ${payload.depot || 'N/A'} · Créée le ${payload.created_at ? new Date(payload.created_at).toLocaleString('fr-FR') : '—'}`;
                details.textContent = payload.details || 'Aucun détail.';
                modal.style.display = 'flex';
            });
        });

        closeModal.addEventListener('click', () => {
            modal.style.display = 'none';
        });
        modal.addEventListener('click', e => {
            if (e.target === modal) {
                modal.style.display = 'none';
            }
        });
    })();
</script>
