<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../config/database.php';

require_login();

$allowedRoles = ['DG', 'DGA', 'Admin', 'Chef Marketing', 'Agent Marketing', 'Accountant', 'Caissier_Comptable'];
$roleName = $_SESSION['role_name'] ?? '';
if (!in_array($roleName, $allowedRoles, true)) {
    header('Location: /masunzu_bar_hotel/dashboards/unauthorized.php');
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    die("<p style='color:#B00020;text-align:center;padding:20px;'>Connexion BD impossible.</p>");
}

// Get province_id for filtering (cashiers see only their province)
$province_id = null;
if ($roleName === 'Caissier_Comptable') {
    $province_id = $_SESSION['province_id'] ?? null;
    if (!$province_id) {
        $stmtProvince = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
        if ($stmtProvince) {
            $stmtProvince->bind_param("i", $_SESSION['user_id']);
            $stmtProvince->execute();
            $provinceRow = $stmtProvince->get_result()->fetch_assoc();
            $_SESSION['province_id'] = $provinceRow['province_id'] ?? null;
            $province_id = $_SESSION['province_id'];
            $stmtProvince->close();
        }
    }
}

// Handle search
$search_term = isset($_GET['search']) ? trim($_GET['search']) : '';
$customers = [];

// Build query
$where_conditions = ["COALESCE(c.is_active, 1) = 1"];
$params = [];
$param_types = '';

if ($search_term) {
    $where_conditions[] = "(c.full_name LIKE ? OR c.tel LIKE ? OR c.nif LIKE ?)";
    $searchPattern = '%' . $search_term . '%';
    $params[] = $searchPattern;
    $params[] = $searchPattern;
    $params[] = $searchPattern;
    $param_types .= 'sss';
}

// For cashiers, filter by province if they have one
if ($province_id && $roleName === 'Caissier_Comptable') {
    // Get customers from their province (via invoices or direct province assignment if exists)
    // For now, we'll show all active customers but this could be filtered by province if needed
    // Most customers don't have a direct province_id, so we show all
}

$where_clause = 'WHERE ' . implode(' AND ', $where_conditions);

$query = "
    SELECT 
        c.id,
        c.full_name,
        c.nif,
        c.tel,
        c.user_type,
        c.can_pay_by_loan,
        c.loan_limit_bif,
        c.loan_balance_bif,
        c.loan_status,
        COALESCE(GREATEST(
            COALESCE(SUM(CASE WHEN cll.direction = 'debit' THEN cll.amount_bif ELSE 0 END), 0) -
            COALESCE(SUM(CASE WHEN cll.direction = 'credit' THEN cll.amount_bif ELSE 0 END), 0),
            0
        ), c.loan_balance_bif, 0) AS ledger_outstanding
    FROM customer c
    LEFT JOIN customer_loan_ledger cll ON (
        cll.customer_id = c.id
        AND cll.entry_type IN ('loan_sale', 'loan_repayment', 'adjustment')
    )
    {$where_clause}
    GROUP BY c.id, c.full_name, c.nif, c.tel, c.user_type, c.can_pay_by_loan, c.loan_limit_bif, c.loan_balance_bif, c.loan_status
    ORDER BY c.full_name ASC
    LIMIT 500
";

$stmt = $conn->prepare($query);
if ($stmt) {
    if (!empty($params)) {
        $stmt->bind_param($param_types, ...$params);
    }
    $stmt->execute();
    $result = $stmt->get_result();
    $customers = $result->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

include __DIR__ . '/../../includes/header.php';
?>

<div style="padding: 20px; background: #F4F0E4; min-height: calc(100vh - 70px);">
    <div style="max-width: 1200px; margin: 0 auto;">
        <h1 style="color: #4B2F1F; font-size: 32px; margin-bottom: 20px; font-weight: 700;">Liste des Clients</h1>
        
        <!-- Search Form -->
        <div style="background-color: #FFFFFF; padding: 20px; border-radius: 8px; margin-bottom: 20px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
            <form method="GET" action="" style="display: flex; gap: 10px; align-items: end;">
                <div style="flex: 1;">
                    <label for="search" style="display: block; color: #4B2F1F; font-weight: 600; margin-bottom: 5px;">Rechercher</label>
                    <input type="text" name="search" id="search" value="<?php echo htmlspecialchars($search_term, ENT_QUOTES, 'UTF-8'); ?>" 
                           placeholder="Nom, téléphone ou NIF..." 
                           style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; font-size: 14px;">
                </div>
                <div>
                    <button type="submit" style="background-color: #4B2F1F; color: #FFFFFF; padding: 10px 20px; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; height: 42px;">Rechercher</button>
                </div>
                <?php if ($search_term): ?>
                    <div>
                        <a href="?" style="background-color: #6c757d; color: #FFFFFF; padding: 10px 20px; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; text-decoration: none; display: inline-block; height: 42px; line-height: 22px;">Réinitialiser</a>
                    </div>
                <?php endif; ?>
            </form>
        </div>
        
        <!-- Customers Table -->
        <div style="background-color: #FFFFFF; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
            <?php if (empty($customers)): ?>
                <p style="text-align: center; color: #666; padding: 40px;">
                    <?php echo $search_term ? 'Aucun client trouvé pour "' . htmlspecialchars($search_term, ENT_QUOTES, 'UTF-8') . '".' : 'Aucun client trouvé.'; ?>
                </p>
            <?php else: ?>
                <div style="overflow-x: auto;">
                    <table style="width: 100%; border-collapse: collapse;">
                        <thead>
                            <tr style="background-color: #4B2F1F; color: #FFFFFF;">
                                <th style="padding: 12px; text-align: left; font-weight: 600;">Nom</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600;">Téléphone</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600;">NIF</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600;">Type</th>
                                <th style="padding: 12px; text-align: right; font-weight: 600;">Crédit Impayé</th>
                                <th style="padding: 12px; text-align: center; font-weight: 600;">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($customers as $customer): ?>
                                <?php
                                $ledgerOutstanding = (int)($customer['ledger_outstanding'] ?? 0);
                                $loanBalance = (int)($customer['loan_balance_bif'] ?? 0);
                                $outstanding = $ledgerOutstanding > 0 ? $ledgerOutstanding : $loanBalance;
                                ?>
                                <tr style="border-bottom: 1px solid #dee2e6;">
                                    <td style="padding: 12px;">
                                        <strong><?php echo htmlspecialchars($customer['full_name'], ENT_QUOTES, 'UTF-8'); ?></strong>
                                    </td>
                                    <td style="padding: 12px;">
                                        <?php echo htmlspecialchars($customer['tel'] ?? '—', ENT_QUOTES, 'UTF-8'); ?>
                                    </td>
                                    <td style="padding: 12px;">
                                        <?php echo htmlspecialchars($customer['nif'] ?? '—', ENT_QUOTES, 'UTF-8'); ?>
                                    </td>
                                    <td style="padding: 12px;">
                                        <?php 
                                        $userType = $customer['user_type'] ?? 'physical';
                                        echo $userType === 'moral' ? 'Personne Morale' : 'Personne Physique';
                                        ?>
                                    </td>
                                    <td style="padding: 12px; text-align: right; <?php echo $outstanding > 0 ? 'color: #dc3545; font-weight: 600;' : 'color: #28a745;'; ?>">
                                        <?php echo number_format($outstanding, 0, ',', '.'); ?> BIF
                                    </td>
                                    <td style="padding: 12px; text-align: center;">
                                        <a href="/masunzu_bar_hotel/modules/customers/view_customer.php?id=<?php echo (int)$customer['id']; ?>" 
                                           style="background-color: #4B2F1F; color: #FFFFFF; padding: 6px 12px; border-radius: 4px; text-decoration: none; font-size: 12px; font-weight: 600;">Voir</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <div style="margin-top: 15px; color: #4B2F1F; font-size: 14px; text-align: center;">
                    <strong><?php echo count($customers); ?></strong> client(s) trouvé(s)
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include __DIR__ . '/../../includes/footer.php'; ?>
<?php $conn->close(); ?>







