<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';
require_login();

// Restrict to Caissier_Comptable role
if ($_SESSION['role_name'] !== 'Caissier_Comptable') {
    error_log("Unauthorized access to transaction_history.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/dashboards/operation_dashboard.php");
    exit;
}

// Initialize database connection
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in transaction_history.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Fetch province_id if not set
if (!isset($_SESSION['province_id']) || $_SESSION['province_id'] === '') {
    $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $_SESSION['province_id'] = $user['province_id'] ?? null;
    $stmt->close();
}
$province_id = $_SESSION['province_id'] ?? null;

if ($province_id === null || $province_id === '') {
    error_log("No province_id for user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Error: No province assigned. Contact administrator.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}
$province_id = (int)$province_id;

// Handle date filter
$startDateInput = $_REQUEST['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$endDateInput = $_REQUEST['end_date'] ?? date('Y-m-d');

$startDateObj = DateTimeImmutable::createFromFormat('Y-m-d', $startDateInput) ?: new DateTimeImmutable('-30 days');
$endDateObj = DateTimeImmutable::createFromFormat('Y-m-d', $endDateInput) ?: new DateTimeImmutable();

if ($startDateObj > $endDateObj) {
    [$startDateObj, $endDateObj] = [$endDateObj, $startDateObj];
}

$rangeStart = $startDateObj->setTime(0, 0, 0)->format('Y-m-d H:i:s');
$rangeEnd = $endDateObj->setTime(23, 59, 59)->format('Y-m-d H:i:s');

function formatPaymentMethod(?string $method): string
{
    if ($method === null || trim($method) === '') {
        return 'Cash';
    }
    $normalized = strtolower($method);
    $normalized = str_replace('_', ' ', $normalized);
    return ucwords($normalized);
}

/**
 * Calculate the cashier's available cash using cashier_account ledger (new method).
 * Falls back to old calculation if table doesn't exist yet.
 */
function getCashierActualCashBalance(mysqli $conn, int $cashierId): float
{
    // Ensure cashier_account table exists
    $createCashierAccountSql = "
        CREATE TABLE IF NOT EXISTS cashier_account (
            id INT AUTO_INCREMENT PRIMARY KEY,
            cashier_id INT NOT NULL,
            shift_id INT NULL,
            transaction_type ENUM('SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT') NOT NULL,
            direction ENUM('CREDIT','DEBIT') NOT NULL,
            amount BIGINT(20) NOT NULL,
            reference_type ENUM('SHIFT','INVOICE','LOAN_PAYMENT','REQUISITION','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','MANUAL') NOT NULL,
            reference_id INT NULL,
            description VARCHAR(255) DEFAULT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            created_by INT NULL,
            INDEX idx_cashier_created (cashier_id, created_at),
            INDEX idx_shift (shift_id),
            INDEX idx_reference (reference_type, reference_id),
            CONSTRAINT fk_ca_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE,
            CONSTRAINT fk_ca_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE SET NULL,
            CONSTRAINT fk_ca_creator FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ";
    @$conn->query($createCashierAccountSql);
    
    // Try new calculation first
    $stmt = $conn->prepare("
        SELECT 
            COALESCE(SUM(CASE WHEN direction = 'CREDIT' THEN amount ELSE 0 END), 0) AS total_credits,
            COALESCE(SUM(CASE WHEN direction = 'DEBIT' THEN amount ELSE 0 END), 0) AS total_debits
        FROM cashier_account
        WHERE cashier_id = ?
    ");
    if ($stmt) {
        $stmt->bind_param("i", $cashierId);
        if ($stmt->execute()) {
            $result = $stmt->get_result()->fetch_assoc();
            $totalCredits = (float)($result['total_credits'] ?? 0);
            $totalDebits = (float)($result['total_debits'] ?? 0);
            $stmt->close();
            $actualCash = $totalCredits - $totalDebits;
            if ($actualCash < 0) {
                $actualCash = 0.0;
            }
            return $actualCash;
        }
        $stmt->close();
    }
    
    // Fallback to old calculation if new method fails
    $cumulativeFromClosed = 0.0;
    $stmt = $conn->prepare("SELECT COALESCE(SUM(end_balance), 0) AS total_closed_balance FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NOT NULL");
    if ($stmt) {
        $stmt->bind_param("i", $cashierId);
        $stmt->execute();
        $cumulativeFromClosed = (float)($stmt->get_result()->fetch_assoc()['total_closed_balance'] ?? 0);
        $stmt->close();
    }

    $currentBalance = 0.0;
    $stmt = $conn->prepare("SELECT shift_start, beginning_balance, deposits FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL ORDER BY shift_start DESC LIMIT 1");
    if ($stmt) {
        $stmt->bind_param("i", $cashierId);
        $stmt->execute();
        $activeShift = $stmt->get_result()->fetch_assoc() ?: [];
        $stmt->close();

        $shiftStart = $activeShift['shift_start'] ?? null;
        if ($shiftStart) {
            $beginningBalance = (float)($activeShift['beginning_balance'] ?? 0);
            $shiftDeposits = (float)($activeShift['deposits'] ?? 0);

            $shiftExpenses = 0.0;
            $stmtExpense = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_expenses FROM expense WHERE cashier_id = ? AND created_at >= ?");
            if ($stmtExpense) {
                $stmtExpense->bind_param("is", $cashierId, $shiftStart);
                $stmtExpense->execute();
                $shiftExpenses = (float)($stmtExpense->get_result()->fetch_assoc()['total_expenses'] ?? 0);
                $stmtExpense->close();
            }

            $shiftTransfers = 0.0;
            $stmtTransfer = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_transfers FROM cash_transfer WHERE cashier_id = ? AND transfer_date >= ?");
            if ($stmtTransfer) {
                $stmtTransfer->bind_param("is", $cashierId, $shiftStart);
                $stmtTransfer->execute();
                $shiftTransfers = (float)($stmtTransfer->get_result()->fetch_assoc()['total_transfers'] ?? 0);
                $stmtTransfer->close();
            }

            $currentBalance = $beginningBalance + $shiftDeposits - $shiftExpenses - $shiftTransfers;
            if ($currentBalance < 0) {
                $currentBalance = 0.0;
            }
        }
    }

    $totalBankDeposits = 0.0;
    $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_deposits FROM bank_deposit WHERE user_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $cashierId);
        $stmt->execute();
        $totalBankDeposits = (float)($stmt->get_result()->fetch_assoc()['total_deposits'] ?? 0);
        $stmt->close();
    }

    $totalTransfersReceived = 0.0;
    $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_received FROM cash_transfer WHERE cashier_id = ? AND status = 'received'");
    if ($stmt) {
        $stmt->bind_param("i", $cashierId);
        $stmt->execute();
        $totalTransfersReceived = (float)($stmt->get_result()->fetch_assoc()['total_received'] ?? 0);
        $stmt->close();
    }

    $actualCash = $cumulativeFromClosed + $currentBalance - $totalBankDeposits - $totalTransfersReceived;
    if ($actualCash < 0) {
        $actualCash = 0.0;
    }
    return $actualCash;
}

/**
 * Fetch cashier transactions across sales, invoices, credit payments, and expenses.
 *
 * @return array<int,array<string,mixed>>
 */
function fetchCaissier_ComptableTransactions(mysqli $conn, int $cashierId, string $startDateTime, string $endDateTime): array
{
    $transactions = [];
    
    // First, try to get transactions from cashier_account ledger (new method)
    $stmt = $conn->prepare("
        SELECT 
            ca.id,
            ca.transaction_type,
            ca.direction,
            ca.amount,
            ca.reference_type,
            ca.reference_id,
            ca.description,
            ca.created_at,
            ca.shift_id
        FROM cashier_account ca
        WHERE ca.cashier_id = ? AND ca.created_at BETWEEN ? AND ?
        ORDER BY ca.created_at ASC
    ");
    if ($stmt) {
        $stmt->bind_param('iss', $cashierId, $startDateTime, $endDateTime);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $amount = (float)($row['amount'] ?? 0);
                if ($amount <= 0 && $row['transaction_type'] !== 'INVOICE_PAYMENT') {
                    continue; // Skip zero-amount transactions except loan sales
                }
                
                $reference = $row['description'] ?? '';
                if ($reference === '') {
                    $reference = $row['transaction_type'] . '-' . (int)$row['id'];
                }
                
                // Map transaction types to categories
                $categoryMap = [
                    'SHIFT_START' => 'Démarrage Session',
                    'INVOICE_PAYMENT' => 'Paiement Commande',
                    'LOAN_REPAYMENT' => 'Remboursement Crédit',
                    'REQUISITION_PAYMENT' => 'Paiement Réquisition',
                    'EXPENSE' => 'Dépense',
                    'BANK_DEPOSIT' => 'Dépôt Bancaire',
                    'CASH_TRANSFER' => 'Transfert Espèces',
                    'ADJUSTMENT' => 'Ajustement'
                ];
                $category = $categoryMap[$row['transaction_type']] ?? $row['transaction_type'];
                
                // Get actual payment method from source table
                $paymentMethod = 'N/A';
                $referenceType = $row['reference_type'] ?? '';
                $referenceId = (int)($row['reference_id'] ?? 0);
                $transactionType = $row['transaction_type'] ?? '';
                
                // Some transaction types don't have payment methods
                if (in_array($transactionType, ['SHIFT_START', 'SHIFT_END', 'BANK_DEPOSIT', 'CASH_TRANSFER', 'ADJUSTMENT'], true)) {
                    $paymentMethod = 'N/A';
                } elseif ($referenceId > 0) {
                    if ($referenceType === 'INVOICE') {
                        // Try paid_invoices first, then order table
                        $stmtPm = $conn->prepare("SELECT payment_method FROM paid_invoices WHERE id = ? LIMIT 1");
                        if ($stmtPm) {
                            $stmtPm->bind_param('i', $referenceId);
                            if ($stmtPm->execute()) {
                                $pmRow = $stmtPm->get_result()->fetch_assoc();
                                if ($pmRow && !empty($pmRow['payment_method'])) {
                                    $paymentMethod = formatPaymentMethod($pmRow['payment_method']);
                                }
                            }
                            $stmtPm->close();
                        }
                        // If not found in paid_invoices, try order table
                        if ($paymentMethod === 'N/A') {
                            $stmtPm = $conn->prepare("SELECT payment_method FROM `order` WHERE id = ? LIMIT 1");
                            if ($stmtPm) {
                                $stmtPm->bind_param('i', $referenceId);
                                if ($stmtPm->execute()) {
                                    $pmRow = $stmtPm->get_result()->fetch_assoc();
                                    if ($pmRow && !empty($pmRow['payment_method'])) {
                                        $paymentMethod = formatPaymentMethod($pmRow['payment_method']);
                                    }
                                }
                                $stmtPm->close();
                            }
                        }
                    } elseif ($referenceType === 'LOAN_PAYMENT') {
                        $stmtPm = $conn->prepare("SELECT payment_method FROM loan_payment WHERE id = ? LIMIT 1");
                        if ($stmtPm) {
                            $stmtPm->bind_param('i', $referenceId);
                            if ($stmtPm->execute()) {
                                $pmRow = $stmtPm->get_result()->fetch_assoc();
                                if ($pmRow && !empty($pmRow['payment_method'])) {
                                    $paymentMethod = formatPaymentMethod($pmRow['payment_method']);
                                }
                            }
                            $stmtPm->close();
                        }
                    } elseif ($referenceType === 'REQUISITION' || $referenceType === 'EXPENSE') {
                        $stmtPm = $conn->prepare("SELECT payment_method FROM expense WHERE id = ? LIMIT 1");
                        if ($stmtPm) {
                            $stmtPm->bind_param('i', $referenceId);
                            if ($stmtPm->execute()) {
                                $pmRow = $stmtPm->get_result()->fetch_assoc();
                                if ($pmRow && !empty($pmRow['payment_method'])) {
                                    $paymentMethod = formatPaymentMethod($pmRow['payment_method']);
                                }
                            }
                            $stmtPm->close();
                        }
                    } elseif ($referenceType === 'BANK_DEPOSIT' || $referenceType === 'CASH_TRANSFER') {
                        // These don't have payment methods, use default
                        $paymentMethod = 'N/A';
                    }
                }
                
                // Fallback: try to extract from description if still N/A
                if ($paymentMethod === 'N/A') {
                    if (stripos($reference, 'Espèces') !== false || stripos($reference, 'cash') !== false) {
                        $paymentMethod = 'Espèces';
                    } elseif (stripos($reference, 'Mobile') !== false || stripos($reference, 'mobile') !== false) {
                        $paymentMethod = 'Mobile';
                    } elseif (stripos($reference, 'Banque') !== false || stripos($reference, 'bank') !== false) {
                        $paymentMethod = 'Banque';
                    } elseif (stripos($reference, 'Crédit') !== false || stripos($reference, 'loan') !== false) {
                        $paymentMethod = 'Crédit';
                    }
                }
                
                $isLoanInvoice = ($row['transaction_type'] === 'INVOICE_PAYMENT' && $amount == 0);
                
                $transactions[] = [
                    'timestamp' => (string)$row['created_at'],
                    'reference' => $reference,
                    'description' => $reference,
                    'amount' => round($amount, 2),
                    'direction' => $row['direction'],
                    'category' => $category,
                    'payment_method' => $paymentMethod,
                    'source' => 'cashier_account',
                    'is_loan_invoice' => $isLoanInvoice,
                    'ignore_for_balance' => $isLoanInvoice,
                ];
            }
        }
        $stmt->close();
        
        // If we got transactions from cashier_account, return them (new method)
        if (!empty($transactions)) {
            return $transactions;
        }
    }
    
    // Fallback to old method if cashier_account is empty or doesn't exist

    // Direct sales (orders)
    $sqlOrders = "SELECT o.id, o.receipt_number, o.invoice_number, o.customer_name, o.paid_amount, o.payment_method, o.sale_type, o.order_type, o.created_at
                  FROM `order` o
                  WHERE o.created_by = ? AND o.created_at BETWEEN ? AND ? AND o.paid_amount IS NOT NULL AND o.paid_amount > 0";
    if ($stmt = $conn->prepare($sqlOrders)) {
        $stmt->bind_param('iss', $cashierId, $startDateTime, $endDateTime);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $amount = (float)($row['paid_amount'] ?? 0);
                if ($amount <= 0) {
                    continue;
                }
                $reference = (string)($row['receipt_number'] ?? '');
                if ($reference === '') {
                    $reference = (string)($row['invoice_number'] ?? '');
                }
                if ($reference === '') {
                    $reference = 'ORDER-' . (int)$row['id'];
                }
                $customerName = trim((string)($row['customer_name'] ?? ''));
                if ($customerName === '' || $customerName === '0') {
                    $customerName = 'Client de passage';
                }
                $category = ($row['sale_type'] === 'wholesale' || $row['order_type'] === 'wholesale')
                    ? 'Vente en Gros'
                    : 'Vente au Détail';
                $paymentMethod = formatPaymentMethod($row['payment_method'] ?? null);

                $transactions[] = [
                    'timestamp' => (string)$row['created_at'],
                    'reference' => $reference,
                    'description' => $customerName,
                    'amount' => round($amount, 2),
                    'direction' => 'CREDIT',
                    'category' => $category,
                    'payment_method' => $paymentMethod,
                    'source' => 'order',
                ];
            }
        } else {
            error_log('Failed to fetch cashier order transactions: ' . $stmt->error);
        }
        $stmt->close();
    } else {
        error_log('Prepare failed for cashier order transactions: ' . $conn->error);
    }

    // Invoice payments (paid_invoices)
    $sqlInvoices = "SELECT p.id, p.invoice_number, p.customer_name, p.paid_amount, p.payment_method, p.sale_type, p.paid_at, p.created_at, p.is_loan_sale
                    FROM paid_invoices p
                    WHERE p.created_by = ? AND p.paid_at BETWEEN ? AND ? AND p.paid_amount IS NOT NULL AND p.paid_amount > 0";
    if ($stmt = $conn->prepare($sqlInvoices)) {
        $stmt->bind_param('iss', $cashierId, $startDateTime, $endDateTime);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $amount = (float)($row['paid_amount'] ?? 0);
                if ($amount <= 0) {
                    continue;
                }
                $reference = (string)($row['invoice_number'] ?? '');
                if ($reference === '') {
                    $reference = 'PAIDINV-' . (int)$row['id'];
                }
                $customerName = trim((string)($row['customer_name'] ?? ''));
                if ($customerName === '') {
                    $customerName = 'Client de commande';
                }
                $category = ($row['sale_type'] === 'wholesale')
                    ? 'Paiement Commande Gros'
                    : 'Paiement Commande';
                $paymentMethod = formatPaymentMethod($row['payment_method'] ?? null);
                $timestamp = (string)($row['paid_at'] ?? '');
                if ($timestamp === '') {
                    $timestamp = (string)$row['created_at'];
                }
                $isLoanInvoice = !empty($row['is_loan_sale']);

                $transactions[] = [
                    'timestamp' => $timestamp,
                    'reference' => $reference,
                    'description' => $customerName,
                    'amount' => round($amount, 2),
                    'direction' => 'CREDIT',
                    'category' => $category,
                    'payment_method' => $paymentMethod,
                    'source' => 'paid_invoices',
                    'is_loan_invoice' => $isLoanInvoice,
                    'ignore_for_balance' => $isLoanInvoice,
                ];
            }
        } else {
            error_log('Failed to fetch cashier invoice transactions: ' . $stmt->error);
        }
        $stmt->close();
    } else {
        error_log('Prepare failed for cashier invoice transactions: ' . $conn->error);
    }

    // Loan repayments
    $sqlLoan = "SELECT lp.id, lp.payment_amount, lp.payment_method, lp.payment_date, c.full_name
                FROM loan_payment lp
                LEFT JOIN customer c ON lp.customer_id = c.id
                WHERE lp.created_by = ? AND lp.payment_date BETWEEN ? AND ?";
    if ($stmt = $conn->prepare($sqlLoan)) {
        $stmt->bind_param('iss', $cashierId, $startDateTime, $endDateTime);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $amount = (float)($row['payment_amount'] ?? 0);
                if ($amount <= 0) {
                    continue;
                }
                $reference = 'LOAN-' . (int)$row['id'];
                $customerName = trim((string)($row['full_name'] ?? 'Client crédit'));
                $paymentMethod = formatPaymentMethod($row['payment_method'] ?? null);
                $timestamp = (string)($row['payment_date'] ?? '');
                if ($timestamp === '') {
                    $timestamp = date('Y-m-d H:i:s');
                }

                $transactions[] = [
                    'timestamp' => $timestamp,
                    'reference' => $reference,
                    'description' => 'Remboursement crédit de ' . $customerName,
                    'amount' => round($amount, 2),
                    'direction' => 'CREDIT',
                    'category' => 'Remboursement Crédit',
                    'payment_method' => $paymentMethod,
                    'source' => 'loan_payment',
                ];
            }
        } else {
            error_log('Failed to fetch loan payments for cashier: ' . $stmt->error);
        }
        $stmt->close();
    } else {
        error_log('Prepare failed for loan payments query: ' . $conn->error);
    }

    // Expenses
    $sqlExpense = "SELECT e.id, e.amount, e.receipt_number, e.payment_method, e.requisition_id, e.created_at
                   FROM expense e
                   WHERE e.cashier_id = ? AND e.created_at BETWEEN ? AND ?";
    if ($stmt = $conn->prepare($sqlExpense)) {
        $stmt->bind_param('iss', $cashierId, $startDateTime, $endDateTime);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $amount = (float)($row['amount'] ?? 0);
                if ($amount <= 0) {
                    continue;
                }
                $reference = (string)($row['receipt_number'] ?? '');
                if ($reference === '') {
                    $reference = 'EXP-' . (int)$row['id'];
                }
                $descriptionParts = [];
                if (!empty($row['requisition_id'])) {
                    $descriptionParts[] = 'Réquisition #' . (int)$row['requisition_id'];
                }
                $descriptionParts[] = 'Reçu dépense ' . $reference;
                $paymentMethod = formatPaymentMethod($row['payment_method'] ?? null);

                $transactions[] = [
                    'timestamp' => (string)$row['created_at'],
                    'reference' => $reference,
                    'description' => implode(' - ', $descriptionParts),
                    'amount' => round($amount, 2),
                    'direction' => 'DEBIT',
                    'category' => 'Dépense',
                    'payment_method' => $paymentMethod,
                    'source' => 'expense',
                ];
            }
        } else {
            error_log('Failed to fetch cashier expenses: ' . $stmt->error);
        }
        $stmt->close();
    } else {
        error_log('Prepare failed for cashier expense query: ' . $conn->error);
    }

    return $transactions;
}

$cashierId = (int)($_SESSION['user_id'] ?? 0);
$transactions = fetchCaissier_ComptableTransactions($conn, $cashierId, $rangeStart, $rangeEnd);

// Sort transactions chronologically (oldest to newest) for running balance calculation
usort($transactions, static function (array $a, array $b): int {
    return strcmp($a['timestamp'], $b['timestamp']);
});

// Calculate starting balance using cashier_account ledger (new method)
// This is more accurate than the old method
$starting_balance = 0.00;
$stmt = $conn->prepare("
    SELECT 
        COALESCE(SUM(CASE WHEN direction = 'CREDIT' THEN amount ELSE 0 END), 0) AS total_credits,
        COALESCE(SUM(CASE WHEN direction = 'DEBIT' THEN amount ELSE 0 END), 0) AS total_debits
    FROM cashier_account
    WHERE cashier_id = ? AND created_at < ?
");
if ($stmt) {
    $stmt->bind_param("is", $_SESSION['user_id'], $rangeStart);
    if ($stmt->execute()) {
        $result = $stmt->get_result()->fetch_assoc();
        $totalCredits = (float)($result['total_credits'] ?? 0);
        $totalDebits = (float)($result['total_debits'] ?? 0);
        $starting_balance = $totalCredits - $totalDebits;
        if ($starting_balance < 0) {
            $starting_balance = 0.00;
        }
    }
    $stmt->close();
} else {
    // Fallback to old method if cashier_account doesn't exist
    $starting_balance_before_range = 0.00;
    $stmt = $conn->prepare("
        SELECT COALESCE(SUM(end_balance), 0) AS total_closed_balance 
        FROM cashier_balance 
        WHERE cashier_id = ? AND shift_end IS NOT NULL AND shift_end < ?
    ");
    if ($stmt) {
        $stmt->bind_param("is", $_SESSION['user_id'], $rangeStart);
        $stmt->execute();
        $starting_balance_before_range = (float)($stmt->get_result()->fetch_assoc()['total_closed_balance'] ?? 0);
        $stmt->close();
    }

    $deposits_before_range = 0.00;
    $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_deposits FROM bank_deposit WHERE user_id = ? AND deposit_date < ?");
    if ($stmt) {
        $stmt->bind_param("is", $_SESSION['user_id'], $rangeStart);
        $stmt->execute();
        $deposits_before_range = (float)($stmt->get_result()->fetch_assoc()['total_deposits'] ?? 0);
        $stmt->close();
    }

    $transfers_before_range = 0.00;
    $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_received FROM cash_transfer WHERE cashier_id = ? AND status = 'received' AND received_at < ?");
    if ($stmt) {
        $stmt->bind_param("is", $_SESSION['user_id'], $rangeStart);
        $stmt->execute();
        $transfers_before_range = (float)($stmt->get_result()->fetch_assoc()['total_received'] ?? 0);
        $stmt->close();
    }

    $starting_balance = $starting_balance_before_range - $deposits_before_range - $transfers_before_range;
    if ($starting_balance < 0) {
        $starting_balance = 0.00;
    }
}

// Calculate running balances (before and after each transaction)
$running_balance = $starting_balance;
foreach ($transactions as &$txn) {
    $txn['balance_before'] = $running_balance;
    if (empty($txn['ignore_for_balance'])) {
        if ($txn['direction'] === 'CREDIT') {
            $running_balance += $txn['amount'];
        } else {
            $running_balance -= $txn['amount'];
        }
        if ($running_balance < 0) {
            $running_balance = 0.00;
        }
    }
    $txn['balance_after'] = $running_balance;
}
unset($txn); // Unset reference

// Sort back to newest first for display
usort($transactions, static function (array $a, array $b): int {
    return strcmp($b['timestamp'], $a['timestamp']);
});

$filteredCredits = 0.0;
$filteredDebits = 0.0;
foreach ($transactions as $txn) {
    if (!empty($txn['ignore_for_balance'])) {
        continue;
    }
    if ($txn['direction'] === 'CREDIT') {
        $filteredCredits += $txn['amount'];
    } else {
        $filteredDebits += $txn['amount'];
    }
}
$filteredNet = $filteredCredits - $filteredDebits;

$todayStartObj = new DateTimeImmutable('today');
$todayEndObj = $todayStartObj->setTime(23, 59, 59);
$todayStart = $todayStartObj->format('Y-m-d H:i:s');
$todayEnd = $todayEndObj->format('Y-m-d H:i:s');

$today_transactions = fetchCaissier_ComptableTransactions($conn, $cashierId, $todayStart, $todayEnd);

// Sort today's transactions chronologically (oldest to newest) for running balance calculation
usort($today_transactions, static function (array $a, array $b): int {
    return strcmp($a['timestamp'], $b['timestamp']);
});

$todayCredits = 0.0;
$todayDebits = 0.0;
foreach ($today_transactions as $txn) {
    if (!empty($txn['ignore_for_balance'])) {
        continue;
    }
    if ($txn['direction'] === 'CREDIT') {
        $todayCredits += $txn['amount'];
    } else {
        $todayDebits += $txn['amount'];
    }
}
$todayNet = $todayCredits - $todayDebits;

// Calculate today's starting balance (balance at start of today)
// = starting balance + net from transactions between rangeStart and todayStart
$today_starting_balance = $starting_balance;
// Add transactions from start of date range to start of today (if rangeStart < todayStart)
if ($rangeStart < $todayStart) {
    $interim_transactions = fetchCaissier_ComptableTransactions($conn, $cashierId, $rangeStart, $todayStart);
    foreach ($interim_transactions as $interim_txn) {
        if (!empty($interim_txn['ignore_for_balance'])) {
            continue;
        }
        if ($interim_txn['direction'] === 'CREDIT') {
            $today_starting_balance += $interim_txn['amount'];
        } else {
            $today_starting_balance -= $interim_txn['amount'];
        }
    }
    if ($today_starting_balance < 0) {
        $today_starting_balance = 0.00;
    }
}

// Calculate running balances for today's transactions
$today_running_balance = $today_starting_balance;
foreach ($today_transactions as &$today_txn) {
    $today_txn['balance_before'] = $today_running_balance;
    if (empty($today_txn['ignore_for_balance'])) {
        if ($today_txn['direction'] === 'CREDIT') {
            $today_running_balance += $today_txn['amount'];
        } else {
            $today_running_balance -= $today_txn['amount'];
        }
        if ($today_running_balance < 0) {
            $today_running_balance = 0.00;
        }
    }
    $today_txn['balance_after'] = $today_running_balance;
}
unset($today_txn); // Unset reference

// Sort back to newest first for display
usort($today_transactions, static function (array $a, array $b): int {
    return strcmp($b['timestamp'], $a['timestamp']);
});

$actual_cash_in_account = getCashierActualCashBalance($conn, $cashierId);

$conn->close();

require_once __DIR__ . '/../../includes/header.php';
?>
<div class="dashboard-container" style="background-color: #F4F0E4; min-height: 100vh; padding: 0; width: 100%;">
    <div class="dashboard-content" style="background-color: #F4F0E4; padding: 40px; border-radius: 0; box-shadow: none; width: 100%; max-width:none;">
        <h2 style="color: #4B2F1F; margin-bottom: 10px;">Historique des Transactions</h2>
        <p style="color: #000000; font-size: 16px;">Voir toutes les transactions pour <?php echo htmlspecialchars($_SESSION['full_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?>.</p>

        <!-- Date Filter Form -->
        <form method="GET" action="" style="margin-bottom: 20px; display: flex; flex-wrap: wrap; gap: 10px; align-items: center;">
            <label style="color: #4B2F1F; font-size: 16px;">Date de Début:</label>
            <input type="date" name="start_date" value="<?php echo htmlspecialchars($startDateObj->format('Y-m-d'), ENT_QUOTES, 'UTF-8'); ?>" style="padding: 8px; border: 2px solid #4B2F1F; border-radius: 5px;">
            <label style="color: #4B2F1F; font-size: 16px;">Date de Fin:</label>
            <input type="date" name="end_date" value="<?php echo htmlspecialchars($endDateObj->format('Y-m-d'), ENT_QUOTES, 'UTF-8'); ?>" style="padding: 8px; border: 2px solid #4B2F1F; border-radius: 5px;">
            <button type="submit" style="padding: 8px 16px; background-color: #F4A261; color: #000000; border: none; border-radius: 5px; cursor: pointer;">Filtrer</button>
        </form>

        <!-- Filtered Transactions -->
        <h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px;">Transactions Filtrées</h3>
        <div style="background-color: #F4F0E4; border-radius: 8px; padding: 15px; margin-bottom: 15px; border-left: 4px solid #4B2F1F;">
            <p style="color: #000000; font-size: 16px; margin: 5px 0;">
                <strong>Solde Initial:</strong> <?php echo number_format((float)$starting_balance, 0, ',', '.') . ' BIF'; ?><br>
                <strong>Total Crédits:</strong> <?php echo number_format((float)$filteredCredits, 0, ',', '.') . ' BIF'; ?><br>
                <strong>Total Débits:</strong> <?php echo number_format((float)$filteredDebits, 0, ',', '.') . ' BIF'; ?><br>
                <strong>Mouvement Net:</strong> <?php echo number_format((float)$filteredNet, 0, ',', '.') . ' BIF'; ?><br>
                <strong style="color: #4B2F1F; font-size: 18px;">Solde Final (Argent Disponible):</strong> <span style="color: #4B2F1F; font-size: 18px; font-weight: 700;"><?php echo number_format((float)$actual_cash_in_account, 0, ',', '.') . ' BIF'; ?></span><br>
                <small style="color: #4B2F1F; font-size: 12px;">Correspond à l'ARGENT DISPONIBLE affiché sur le tableau de bord caissier.</small>
            </p>
        </div>
        <?php if (empty($transactions)): ?>
            <p style="color: #000000; font-size: 16px;">Aucune transaction trouvée pour la période sélectionnée.</p>
        <?php else: ?>
            <table style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden; margin-bottom: 20px;">
                        <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 12px; text-align: left;">Date &amp; Heure</th>
                        <th style="padding: 12px; text-align: left;">Type</th>
                        <th style="padding: 12px; text-align: left;">Référence</th>
                        <th style="padding: 12px; text-align: left;">Détails</th>
                        <th style="padding: 12px; text-align: left;">Méthode de Paiement</th>
                        <th style="padding: 12px; text-align: left;">Direction</th>
                        <th style="padding: 12px; text-align: right;">Montant (BIF)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($transactions as $txn): ?>
                        <tr style="background-color: <?php echo $txn['direction'] === 'CREDIT' ? '#E6FFE6' : '#FFE6E6'; ?>;">
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($txn['timestamp'])), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars($txn['category'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars($txn['reference'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars($txn['description'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars($txn['payment_method'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars(ucfirst(strtolower($txn['direction'])), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: right; font-weight: <?php echo $txn['direction'] === 'CREDIT' ? '600' : '600'; ?>; color: <?php echo $txn['direction'] === 'CREDIT' ? '#2E7D32' : '#C62828'; ?>;">
                                <?php echo $txn['direction'] === 'CREDIT' ? '+' : '-'; ?><?php echo number_format((float)$txn['amount'], 0, ',', '.') . ' BIF'; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>

        <!-- Today's Transactions -->
        <h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px;">Transactions d'Aujourd'hui (<?php echo date('Y-m-d'); ?>)</h3>
        <div style="background-color: #F4F0E4; border-radius: 8px; padding: 15px; margin-bottom: 15px; border-left: 4px solid #2A9D8F;">
            <p style="color: #000000; font-size: 16px; margin: 5px 0;">
                <strong>Solde Initial:</strong> <?php echo number_format((float)$today_starting_balance, 0, ',', '.') . ' BIF'; ?><br>
                <strong>Total Crédits:</strong> <?php echo number_format((float)$todayCredits, 0, ',', '.') . ' BIF'; ?><br>
                <strong>Total Débits:</strong> <?php echo number_format((float)$todayDebits, 0, ',', '.') . ' BIF'; ?><br>
                <strong>Mouvement Net:</strong> <?php echo number_format((float)$todayNet, 0, ',', '.') . ' BIF'; ?><br>
                <strong style="color: #2A9D8F; font-size: 18px;">Solde Final:</strong> <span style="color: #2A9D8F; font-size: 18px; font-weight: 700;"><?php echo number_format((float)$today_running_balance, 0, ',', '.') . ' BIF'; ?></span>
            </p>
        </div>
        <?php if (empty($today_transactions)): ?>
            <p style="color: #000000; font-size: 16px;">Aucune transaction trouvée pour aujourd'hui.</p>
        <?php else: ?>
            <table style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden; margin-bottom: 20px;">
                        <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 12px; text-align: left;">Date &amp; Heure</th>
                        <th style="padding: 12px; text-align: left;">Type</th>
                        <th style="padding: 12px; text-align: left;">Référence</th>
                        <th style="padding: 12px; text-align: left;">Détails</th>
                        <th style="padding: 12px; text-align: left;">Méthode de Paiement</th>
                        <th style="padding: 12px; text-align: left;">Direction</th>
                        <th style="padding: 12px; text-align: right;">Montant (BIF)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($today_transactions as $txn): ?>
                        <tr style="background-color: <?php echo $txn['direction'] === 'CREDIT' ? '#E6FFE6' : '#FFE6E6'; ?>;">
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($txn['timestamp'])), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars($txn['category'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars($txn['reference'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars($txn['description'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars($txn['payment_method'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left;"><?php echo htmlspecialchars(ucfirst(strtolower($txn['direction'])), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: right; font-weight: <?php echo $txn['direction'] === 'CREDIT' ? '600' : '600'; ?>; color: <?php echo $txn['direction'] === 'CREDIT' ? '#2E7D32' : '#C62828'; ?>;">
                                <?php echo $txn['direction'] === 'CREDIT' ? '+' : '-'; ?><?php echo number_format((float)$txn['amount'], 0, ',', '.') . ' BIF'; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>

        <div style="margin-top: 20px; text-align: center;">
            <a href="/masunzu_bar_hotel/dashboards/cashier_dashboard.php" style="background-color: #F4A261; color: #000000; padding: 10px 20px; border-radius: 8px; text-decoration: none; font-weight: bold;">Retour au Tableau de Bord</a>
        </div>
    </div>
</div>
<?php require_once __DIR__ . '/../../includes/footer.php'; ?>
