<?php
// Prevent multiple inclusions
if (defined('MASUNZU_AUTH_PHP_LOADED')) {
    error_log("auth.php: Already included, skipping from " . debug_backtrace()[0]['file']);
    return;
}
define('MASUNZU_AUTH_PHP_LOADED', true);
error_log("auth.php: Loaded from " . debug_backtrace()[0]['file']);

// Start session
session_start();
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/config/database.php';
require_once __DIR__ . '/functions.php';

// Enable error reporting
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Check if user is logged in
function is_logged_in(): bool {
    $logged_in = isset($_SESSION['user_id']);
    error_log("is_logged_in: " . ($logged_in ? "true" : "false") . ", user_id: " . ($_SESSION['user_id'] ?? 'unset'));
    return $logged_in;
}

// Redirect to login if not authenticated
function require_login(): void {
    if (!is_logged_in()) {
        error_log("require_login: Redirecting to login.php");
        header("Location: /masunzu_bar_hotel/modules/login/login.php");
        exit;
    }
}

// Log action to database
// Commented out to avoid redeclaration; use log_action from functions.php
/*
function log_action(int $user_id, string $action, string $description): void {
    $conn = get_db_connection();
    if (!$conn) {
        error_log("log_action: Database connection failed");
        return;
    }
    $stmt = $conn->prepare("INSERT INTO `log` (user_id, action, description, created_at) VALUES (?, ?, ?, NOW())");
    $stmt->bind_param("iss", $user_id, $action, $description);
    $stmt->execute();
    $stmt->close();
    $conn->close();
}
*/

/**
 * Ensure the user_session table can accept inserts without providing `id`.
 */
function ensure_user_session_schema(mysqli $conn): void {
    static $schema_checked = false;
    if ($schema_checked) {
        return;
    }

    $dbResult = $conn->query("SELECT DATABASE() AS db_name");
    $dbName = $dbResult ? ($dbResult->fetch_assoc()['db_name'] ?? null) : null;
    if (!$dbName) {
        error_log("ensure_user_session_schema: Unable to determine database name");
        return;
    }

    // Ensure primary key on id.
    if ($stmtPk = $conn->prepare("
        SELECT 1 
        FROM information_schema.TABLE_CONSTRAINTS 
        WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'user_session' AND CONSTRAINT_TYPE = 'PRIMARY KEY' 
        LIMIT 1
    ")) {
        $stmtPk->bind_param("s", $dbName);
        $stmtPk->execute();
        $hasPk = $stmtPk->get_result()->num_rows > 0;
        $stmtPk->close();
        if (!$hasPk) {
            if (!$conn->query("ALTER TABLE user_session ADD PRIMARY KEY (id)")) {
                error_log("ensure_user_session_schema: Failed to add primary key on user_session.id - " . $conn->error);
            }
        }
    } else {
        error_log("ensure_user_session_schema: Unable to prepare PK check - " . $conn->error);
    }

    // Ensure unique session_id to support ON DUPLICATE KEY UPDATE.
    if ($stmtUnique = $conn->prepare("
        SELECT 1 
        FROM information_schema.STATISTICS 
        WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'user_session' AND INDEX_NAME = 'uk_session_id' 
        LIMIT 1
    ")) {
        $stmtUnique->bind_param("s", $dbName);
        $stmtUnique->execute();
        $hasUnique = $stmtUnique->get_result()->num_rows > 0;
        $stmtUnique->close();
        if (!$hasUnique) {
            if (!$conn->query("ALTER TABLE user_session ADD UNIQUE KEY uk_session_id (session_id)")) {
                error_log("ensure_user_session_schema: Failed to add unique index on session_id - " . $conn->error);
            }
        }
    } else {
        error_log("ensure_user_session_schema: Unable to prepare unique index check - " . $conn->error);
    }

    // Ensure id column is AUTO_INCREMENT so inserts without id succeed.
    if ($stmtAuto = $conn->prepare("
        SELECT EXTRA 
        FROM information_schema.COLUMNS 
        WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'user_session' AND COLUMN_NAME = 'id' 
        LIMIT 1
    ")) {
        $stmtAuto->bind_param("s", $dbName);
        $stmtAuto->execute();
        $extra = $stmtAuto->get_result()->fetch_assoc()['EXTRA'] ?? '';
        $stmtAuto->close();

        if (stripos($extra, 'auto_increment') === false) {
            if (!$conn->query("ALTER TABLE user_session MODIFY id int(11) NOT NULL AUTO_INCREMENT")) {
                error_log("ensure_user_session_schema: Failed to set AUTO_INCREMENT on user_session.id - " . $conn->error);
            }
        }
    } else {
        error_log("ensure_user_session_schema: Unable to prepare AUTO_INCREMENT check - " . $conn->error);
    }

    $schema_checked = true;
}

function log_user_session(int $user_id): void {
    $conn = get_db_connection();
    if (!$conn) {
        error_log("log_user_session: Database connection failed");
        return;
    }

    ensure_user_session_schema($conn);

    $session_id = session_id();
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? '';
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    
    $stmt = $conn->prepare("INSERT INTO user_session (user_id, session_id, ip_address, user_agent, logout_time) VALUES (?, ?, ?, ?, NULL) ON DUPLICATE KEY UPDATE user_id = ?, ip_address = ?, user_agent = ?, logout_time = NULL, login_time = NOW()");
    if ($stmt) {
        $stmt->bind_param("isssiss", $user_id, $session_id, $ip_address, $user_agent, $user_id, $ip_address, $user_agent);
        $stmt->execute();
        $stmt->close();
    }
    $conn->close();
}

function logout_user_session(): void {
    $conn = get_db_connection();
    if (!$conn) {
        error_log("logout_user_session: Database connection failed");
        return;
    }
    $session_id = session_id();
    $stmt = $conn->prepare("UPDATE user_session SET logout_time = NOW() WHERE session_id = ? AND logout_time IS NULL");
    if ($stmt) {
        $stmt->bind_param("s", $session_id);
        $stmt->execute();
        $stmt->close();
    }
    $conn->close();
}

// Login function
function login(string $username, string $password): array {
    $conn = get_db_connection();
    if (!$conn) {
        error_log("login: Database connection failed");
        return ['success' => false, 'error' => 'Database connection failed'];
    }
    $stmt = $conn->prepare("SELECT u.id, u.username, u.password, u.full_name, u.role_id, u.province_id, u.is_active, r.name AS role_name FROM user u LEFT JOIN role r ON u.role_id = r.id WHERE u.username = ?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 1) {
        $user = $result->fetch_assoc();
        if ($user['is_active'] === 0) {
            error_log("Login failed: User $username is inactive");
            $stmt->close();
            $conn->close();
            return ['success' => false, 'error' => 'Your account is deactivated. Please contact an administrator.'];
        }
        if (password_verify($password, $user['password'])) {
            $normalizedRoleName = mbh_normalize_role_name($user['role_name']);
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['full_name'] = $user['full_name'];
            $_SESSION['role_id'] = $user['role_id'];
            $_SESSION['role_original_name'] = $user['role_name'];
            $_SESSION['role_name'] = $normalizedRoleName;
            $_SESSION['role_label'] = mbh_get_role_label($normalizedRoleName);
            $_SESSION['role_departments'] = mbh_get_role_departments($normalizedRoleName);
            $_SESSION['province_id'] = $user['province_id'];
            error_log("Login successful for $username. Session: " . print_r($_SESSION, true));
            log_action($user['id'], 'login', "User $username logged in");
            log_user_session($user['id']);
            $stmt->close();
            $conn->close();
            return ['success' => true];
        } else {
            error_log("Login failed: Incorrect password for $username");
        }
    } else {
        error_log("Login failed: User $username not found");
    }
    
    $stmt->close();
    $conn->close();
    return ['success' => false, 'error' => 'Invalid username or password'];
}

// Redirect to role-specific dashboard
function redirect_to_dashboard(): void {
    if (!is_logged_in() || !isset($_SESSION['role_name'])) {
        error_log("redirect_to_dashboard: No valid session data, redirecting to login");
        header("Location: /masunzu_bar_hotel/modules/login/login.php");
        exit;
    }
    error_log("Redirecting user with role: " . $_SESSION['role_name']);
    $role = mbh_normalize_role_name($_SESSION['role_name']);
    $dashboard_map = [
        'Board' => 'admin_dashboard.php',
        'DG' => 'admin_dashboard.php',
        'DGA' => 'admin_dashboard.php',
        'Admin' => 'admin_dashboard.php',
        'Operation Manager' => 'operation_dashboard.php',
        'Procurement Manager' => 'procurement_dashboard.php',
        'Finance Head' => 'finance_dashboard.php',
    'Marketing Head' => 'marketing_dashboard.php',
    'Chef Marketing' => 'marketing_dashboard.php',
    'Chef Marketing' => 'marketing_dashboard.php',
    'Agent Marketing' => 'marketing_dashboard.php',
        'Stock Manager' => 'stock_manager_dashboard.php',
        'Accountant' => 'accountant_dashboard.php',
        'Marketing Agent' => 'marketing_dashboard.php',
        'Cashier' => 'cashier_dashboard.php',
        'Caissier_Comptable' => 'cashier_dashboard.php'
    ];
    $dashboard = $dashboard_map[$role] ?? 'unauthorized.php';
    $dashboard_path = "/masunzu_bar_hotel/dashboards/$dashboard";
    if (!file_exists("/opt/lampp/htdocs$dashboard_path")) {
        error_log("redirect_to_dashboard: Dashboard file $dashboard_path does not exist, redirecting to unauthorized.php");
        $dashboard_path = "/masunzu_bar_hotel/dashboards/unauthorized.php";
    }
    error_log("Redirecting to $dashboard_path");
    header("Location: $dashboard_path");
    exit;
}
?>
